# coding: utf-8


import tensorflow as tf
import tensorflow_datasets as tfds
import numpy as np
import pandas as pd
import os
import gzip
import shutil
from collections import Counter
from tensorflow.keras.layers import Embedding
from tensorflow.keras import Sequential
from tensorflow.keras.layers import SimpleRNN
from tensorflow.keras.layers import Dense
from tensorflow.keras.layers import LSTM
from tensorflow.keras.layers import GRU
from tensorflow.keras.layers import Bidirectional

# Copyright (c) 2019 [Sebastian Raschka](sebastianraschka.com)
# 
# https://github.com/rasbt/python-machine-learning-book-3rd-edition
# 
# [MIT License](https://github.com/rasbt/python-machine-learning-book-3rd-edition/blob/master/LICENSE.txt)

# # Rozdział 16. Modelowanie danych sekwencyjnych za pomocą rekurencyjnych sieci neuronowych (1/2)

# Zwróć uwagę, że rozszerzenie zawierające nieobowiązkowy znak wodny stanowi niewielki plugin notatnika IPython / Jupyter, który zaprojektowałem w celu powielania kodu źródłowego. Wystarczy pominąć poniższe wiersze kodu:









# # Wprowadzenie do danych sekwencyjnych
# 
# ## Modelowanie danych sekwencyjnych — kolejność ma znaczenie
# 
# ## Przedstawianie sekwencji
# 
# 





# ## Różne kategorie modelowania sekwencji





# # Sieci rekurencyjne służące do modelowania sekwencji
# 
# ## Mechanizm zapętlania w sieciach rekurencyjnych
# 









# ## Obliczanie aktywacji w sieciach rekurencyjnych
# 









# ## Rekurencja w warstwie ukrytej a rekurencja w warstwie wyjściowej







tf.random.set_seed(1)

rnn_layer = tf.keras.layers.SimpleRNN(
    units=2, use_bias=True, 
    return_sequences=True)
rnn_layer.build(input_shape=(None, None, 5))

w_xh, w_oo, b_h = rnn_layer.weights

print('Rozmiar W_xh:', w_xh.shape)
print('Rozmiar W_oo:', w_oo.shape)
print('Rozmiar b_h:', b_h.shape)




x_seq = tf.convert_to_tensor(
    [[1.0]*5, [2.0]*5, [3.0]*5],
    dtype=tf.float32)


## wynik z klasy SimpleRNN:
output = rnn_layer(tf.reshape(x_seq, shape=(1, 3, 5)))

## ręczne obliczanie wyniku:
out_man = []
for t in range(len(x_seq)):
    xt = tf.reshape(x_seq[t], (1, 5))
    print('Takt  {} =>'.format(t))
    print('   Wejściowa            :', xt.numpy())
    
    ht = tf.matmul(xt, w_xh) + b_h    
    print('   Ukryta           :', ht.numpy())
    
    if t>0:
        prev_o = out_man[t-1]
    else:
        prev_o = tf.zeros(shape=(ht.shape))
        
    ot = ht + tf.matmul(prev_o, w_oo)
    ot = tf.math.tanh(ot)
    out_man.append(ot)
    print('   Wynik (ręcznie) :', ot.numpy())
    print('   Wynik (SimpleRNN) :'.format(t), output[0][t].numpy())
    print()


# ## Problemy z uczeniem długofalowych oddziaływań
# 





# 
# ## Jednostki LSTM





# # Implementowanie wielowarstwowej sieci rekurencyjnej przy użyciu biblioteki TensorFlow do modelowania sekwencji
# 
# ## Pierwszy projekt — przewidywanie sentymentów na recenzjach z zestawu danych IMDb
# 
# ### Przygotowanie danych recenzji
# 
# 









with gzip.open('../r08/movie_data.csv.gz', 'rb') as f_in, open('movie_data.csv', 'wb') as f_out:
    shutil.copyfileobj(f_in, f_out)




df = pd.read_csv('movie_data.csv', encoding='utf-8')

df.tail()




# Etap 1.: stworzenie zestawu danych

target = df.pop('sentiment')

ds_raw = tf.data.Dataset.from_tensor_slices(
    (df.values, target.values))

## Sprawdzenie:
for ex in ds_raw.take(3):
    tf.print(ex[0].numpy()[0][:50], ex[1])


#  * **Podział na zbiory uczący/walidacyjny/testowy**



tf.random.set_seed(1)

ds_raw = ds_raw.shuffle(
    50000, reshuffle_each_iteration=False)

ds_raw_test = ds_raw.take(25000)
ds_raw_train_valid = ds_raw.skip(25000)
ds_raw_train = ds_raw_train_valid.take(20000)
ds_raw_valid = ds_raw_train_valid.skip(20000)


#  * **Tokenizer i koder**
#    * `tfds.features.text.Tokenizer`: https://www.tensorflow.org/datasets/api_docs/python/tfds/features/text/Tokenizer
#    * `tfds.features.text.TokenTextEncoder`: https://www.tensorflow.org/datasets/api_docs/python/tfds/features/text/TokenTextEncoder

#  * **Kodowanie sekwencji: przechowywanie 100 ostatnich elementów w każdej sekwencji**



## Etap 2.: Wyszukiwanie niepowtarzalnych tokenów (wyrazów)


tokenizer = tfds.features.text.Tokenizer()
token_counts = Counter()

for example in ds_raw_train:
    tokens = tokenizer.tokenize(example[0].numpy()[0])
    token_counts.update(tokens)
    
print('Rozmiar wokabularza:', len(token_counts))




## Etap 3.: Kodowanie tokenów do postaci liczb całkowitych

encoder = tfds.features.text.TokenTextEncoder(token_counts)

example_str = 'This is an example!'
encoder.encode(example_str)




## Etap 3A: Zdefiniowanie funkcji realizującej przekształcenie

def encode(text_tensor, label):
    text = text_tensor.numpy()[0]
    encoded_text = encoder.encode(text)
    return encoded_text, label

## Etap 3B: opakowanie funkcji kodującej do postaci operacji TF
def encode_map_fn(text, label):
    return tf.py_function(encode, inp=[text, label], 
                          Tout=(tf.int64, tf.int64))




ds_train = ds_raw_train.map(encode_map_fn)
ds_valid = ds_raw_valid.map(encode_map_fn)
ds_test = ds_raw_test.map(encode_map_fn)

tf.random.set_seed(1)
for example in ds_train.shuffle(1000).take(5):
    print('Długość sekwencji:', example[0].shape)
    
example


#  * **Porównanie metod batch() i padded_batch()**

# ```python
# 
# # poniższy kod spowoduje wyświetlenie błędu
# 
# 
# BATCH_SIZE = 32
# train_data = all_encoded_data.batch(BATCH_SIZE)
# 
# next(iter(train_data))
# 
# # Powyższy kod spowoduje błąd
# # Nie możemy użyć metody .batch() na tym zestawie danych
# ```



## Pobiera mały podzbiór

ds_subset = ds_train.take(8)
for example in ds_subset:
    print('Rozmiar przykładu:', example[0].shape)

## Dzieli zestaw danych na grupy
ds_batched = ds_subset.padded_batch(
    4, padded_shapes=([-1], []))

for batch in ds_batched:
    print('Rozmiar grupy:', batch[0].shape)




## dzieli zbiory danych na grupy
train_data = ds_train.padded_batch(
    32, padded_shapes=([-1],[]))

valid_data = ds_valid.padded_batch(
    32, padded_shapes=([-1],[]))

test_data = ds_test.padded_batch(
    32, padded_shapes=([-1],[]))


# ### Warstwy wektorów właściwościowych w kodowaniu zdań
# 
# 
#  * `input_dim`: liczba słów, tj. maksymalna wartość indeksu + 1.
#  * `output_dim`: 
#  * `input_length`: długość sekwencji (uzupełnionej zerami)
#     * na przykład, `'This is an example' -> [0, 0, 0, 0, 0, 0, 3, 1, 8, 9]`   
#     => input_lenght ma wartość 10
#  
#  
# 
#  * Podczas wywoływania warstwy na wejściu przyjmowane są wartości całkowite,   
#  warstwa wektorów właściwościowych przekształca każdą wartość całkowitą w wektor wartości zmiennoprzecinkowych o rozmiarze `[output_dim]`
#    * Jeżeli wymiary danych wejściowych są równe `[BATCH_SIZE]`, wymiary danych wyjściowych przyjmą postać `[BATCH_SIZE, output_dim]`
#    * Jeżeli wymiary danych wejściowych są równe `[BATCH_SIZE, 10]`, wymiary danych wyjściowych przyjmą postać `[BATCH_SIZE, 10, output_dim]`









model = tf.keras.Sequential()

model.add(Embedding(input_dim=100,
                    output_dim=6,
                    input_length=20,
                    name='warstwa-osadzania'))

model.summary()


# ### Budowanie modelu sieci rekurencyjnej
# 
# * **Warstwy sieci RNN w interfejsie Keras:**
#   * `tf.keras.layers.SimpleRNN(units, return_sequences=False)`
#   * `tf.keras.layers.LSTM(..)`
#   * `tf.keras.layers.GRU(..)`
#   * `tf.keras.layers.Bidirectional()`
#  
# * **Określanie `return_sequences=?`**
#   * W wielowarstwowej sieci rekurencyjnej, we wszystkich warstwach oprócz ostatniej powinien być zdefiniowany atrybut `return_sequenes=True`
#   * W ostatniej warstwie rekurencyjnej wartość tego atrybutu jest zależna od rodzaju problemu: 
#      * wiele-do-wielu: -> `return_sequences=True`
#      * wiele-do-jednego : -> `return_sequenes=False`
#      * ..
#     



## Przykład tworzenia modelu RNN za pomocą warstwy SimpleRNN


model = Sequential()
model.add(Embedding(1000, 32))
model.add(SimpleRNN(32, return_sequences=True))
model.add(SimpleRNN(32))
model.add(Dense(1))
model.summary()




## przykład tworzenia modelu RNN za pomocą warstwy LSTM




model = Sequential()
model.add(Embedding(10000, 32))
model.add(LSTM(32, return_sequences=True))
model.add(LSTM(32))
model.add(Dense(1))
model.summary()




## przykład tworzenia modelu RNN za pomocą warstwy GRU


model = Sequential()
model.add(Embedding(10000, 32))
model.add(GRU(32, return_sequences=True))
model.add(GRU(32))
model.add(Dense(1))
model.summary()


# ### Uczenie modelu sieci rekurencyjnej przeznaczonego do analizy sentymentów



embedding_dim = 20
vocab_size = len(token_counts) + 2

tf.random.set_seed(1)

## tworzy model
bi_lstm_model = tf.keras.Sequential([
    tf.keras.layers.Embedding(
        input_dim=vocab_size,
        output_dim=embedding_dim,
        name='warstwa-osadzania'),
    
    tf.keras.layers.Bidirectional(
        tf.keras.layers.LSTM(64, name='warstwa-lstm'),
        name='dwukierunkowa-lstm'), 

    tf.keras.layers.Dense(64, activation='relu'),
    
    tf.keras.layers.Dense(1, activation='sigmoid')
])

bi_lstm_model.summary()

## kompilacja i uczenie:
bi_lstm_model.compile(
    optimizer=tf.keras.optimizers.Adam(1e-3),
    loss=tf.keras.losses.BinaryCrossentropy(from_logits=False),
    metrics=['accuracy'])

history = bi_lstm_model.fit(
    train_data, 
    validation_data=valid_data, 
    epochs=10)

## ocena na danych testowych
test_results= bi_lstm_model.evaluate(test_data)
print('Dokładność testowa: {:.2f}%'.format(test_results[1]*100))




if not os.path.exists('modele'):
    os.mkdir('modele')


bi_lstm_model.save('modele/Dwukier-LSTM-pelna-dlug-sekw.h5')


#  * **Sprawdzenie działania modelu SimpleRNN na krótkich sekwencjach**



def preprocess_datasets(
    ds_raw_train, 
    ds_raw_valid, 
    ds_raw_test,
    max_seq_length=None,
    batch_size=32):
    
    ## (Etap 1. został już zrealizowany)
    ## Etap 2.: Wyszukiwanie niepowtarzalnych tokenów
    tokenizer = tfds.features.text.Tokenizer()
    token_counts = Counter()

    for example in ds_raw_train:
        tokens = tokenizer.tokenize(example[0].numpy()[0])
        if max_seq_length is not None:
            tokens = tokens[-max_seq_length:]
        token_counts.update(tokens)

    print('Rozmiar wokabularza:', len(token_counts))


    ## Etap 3.: Kodowanie tekstów
    encoder = tfds.features.text.TokenTextEncoder(token_counts)
    def encode(text_tensor, label):
        text = text_tensor.numpy()[0]
        encoded_text = encoder.encode(text)
        if max_seq_length is not None:
            encoded_text = encoded_text[-max_seq_length:]
        return encoded_text, label

    def encode_map_fn(text, label):
        return tf.py_function(encode, inp=[text, label], 
                              Tout=(tf.int64, tf.int64))

    ds_train = ds_raw_train.map(encode_map_fn)
    ds_valid = ds_raw_valid.map(encode_map_fn)
    ds_test = ds_raw_test.map(encode_map_fn)

    ## Etap 4.: Tworzenie minigrup
    train_data = ds_train.padded_batch(
        batch_size, padded_shapes=([-1],[]))

    valid_data = ds_valid.padded_batch(
        batch_size, padded_shapes=([-1],[]))

    test_data = ds_test.padded_batch(
        batch_size, padded_shapes=([-1],[]))

    return (train_data, valid_data, 
            test_data, len(token_counts))




def build_rnn_model(embedding_dim, vocab_size,
                    recurrent_type='SimpleRNN',
                    n_recurrent_units=64,
                    n_recurrent_layers=1,
                    bidirectional=True):

    tf.random.set_seed(1)

    # tworzy model
    model = tf.keras.Sequential()
    
    model.add(
        Embedding(
            input_dim=vocab_size,
            output_dim=embedding_dim,
            name='warstwa-osadzania')
    )
    
    for i in range(n_recurrent_layers):
        return_sequences = (i < n_recurrent_layers-1)
            
        if recurrent_type == 'SimpleRNN':
            recurrent_layer = SimpleRNN(
                units=n_recurrent_units, 
                return_sequences=return_sequences,
                name='warstwa-simplernn-{}'.format(i))
        elif recurrent_type == 'LSTM':
            recurrent_layer = LSTM(
                units=n_recurrent_units, 
                return_sequences=return_sequences,
                name='warstwa-lstm-{}'.format(i))
        elif recurrent_type == 'GRU':
            recurrent_layer = GRU(
                units=n_recurrent_units, 
                return_sequences=return_sequences,
                name='warstwa-gru-{}'.format(i))
        
        if bidirectional:
            recurrent_layer = Bidirectional(
                recurrent_layer, name='dwukier-'+recurrent_layer.name)
            
        model.add(recurrent_layer)

    model.add(tf.keras.layers.Dense(64, activation='relu'))
    model.add(tf.keras.layers.Dense(1, activation='sigmoid'))
    
    return model






batch_size = 32
embedding_dim = 20
max_seq_length = 100

train_data, valid_data, test_data, n = preprocess_datasets(
    ds_raw_train, ds_raw_valid, ds_raw_test, 
    max_seq_length=max_seq_length, 
    batch_size=batch_size
)


vocab_size = n + 2

rnn_model = build_rnn_model(
    embedding_dim, vocab_size,
    recurrent_type='SimpleRNN', 
    n_recurrent_units=64,
    n_recurrent_layers=1,
    bidirectional=True)

rnn_model.summary()




rnn_model.compile(optimizer=tf.keras.optimizers.Adam(1e-3),
                  loss=tf.keras.losses.BinaryCrossentropy(from_logits=False),
                  metrics=['accuracy'])


history = rnn_model.fit(
    train_data, 
    validation_data=valid_data, 
    epochs=10)




results = rnn_model.evaluate(test_data)




print('Dokładność testowa: {:.2f}%'.format(results[1]*100))


# ## Dodatkowe ćwiczenie: 
# 
# ### Jednokierunkowa sieć SimpleRNN dla pełnych sekwencji



batch_size = 32
embedding_dim = 20
max_seq_length = None

train_data, valid_data, test_data, n = preprocess_datasets(
    ds_raw_train, ds_raw_valid, ds_raw_test, 
    max_seq_length=max_seq_length, 
    batch_size=batch_size
)


vocab_size = n + 2

rnn_model = build_rnn_model(
    embedding_dim, vocab_size,
    recurrent_type='SimpleRNN', 
    n_recurrent_units=64,
    n_recurrent_layers=1,
    bidirectional=False)

rnn_model.summary()




rnn_model.compile(optimizer=tf.keras.optimizers.Adam(1e-3),
                  loss=tf.keras.losses.BinaryCrossentropy(from_logits=False),
                  metrics=['accuracy'])

history = rnn_model.fit(
    train_data, 
    validation_data=valid_data, 
    epochs=10)


# # Dodatek
# 

# ### A) Alternatywny sposób uzyskania zestawu danych: za pomocą tensorflow_datasets



imdb_bldr = tfds.builder('imdb_reviews')
print(imdb_bldr.info)

imdb_bldr.download_and_prepare()

datasets = imdb_bldr.as_dataset(shuffle_files=False)

datasets.keys()




imdb_train = datasets['train']
imdb_train = datasets['test']


# ### B) Tokenizer i koder
# 
#  * `tfds.features.text.Tokenizer`: https://www.tensorflow.org/datasets/api_docs/python/tfds/features/text/Tokenizer
#  * `tfds.features.text.TokenTextEncoder`: https://www.tensorflow.org/datasets/api_docs/python/tfds/features/text/TokenTextEncoder
# 
# 



vocab_set = {'a', 'b', 'c', 'd'}
encoder = tfds.features.text.TokenTextEncoder(vocab_set)
print(encoder)

print(encoder.encode(b'a b c d, , : .'))

print(encoder.encode(b'a b c d e f g h i z'))


# ### C) Wstępne przetwarzanie tekstu w interfejsie Keras 



TOP_K = 200
MAX_LEN = 10

tokenizer = tf.keras.preprocessing.text.Tokenizer(num_words=TOP_K)

tokenizer.fit_on_texts(['this is an example', 'je suis en forme '])
sequences = tokenizer.texts_to_sequences(['this is an example', 'je suis en forme '])
print(sequences)

tf.keras.preprocessing.sequence.pad_sequences(sequences, maxlen=MAX_LEN)




TOP_K = 20000
MAX_LEN = 500

tokenizer = tf.keras.preprocessing.text.Tokenizer(num_words=TOP_K)

tokenizer.fit_on_texts(
    [example['text'].numpy().decode('utf-8') 
     for example in imdb_train])

x_train = tokenizer.texts_to_sequences(
    [example['text'].numpy().decode('utf-8')
     for example in imdb_train])

print(len(x_train))


x_train_padded = tf.keras.preprocessing.sequence.pad_sequences(
    x_train, maxlen=MAX_LEN)

print(x_train_padded.shape)


# ### D) Osadzanie
# 
# 





tf.random.set_seed(1)
embed = Embedding(input_dim=100, output_dim=4)

inp_arr = np.array([1, 98, 5, 6, 67, 45])
tf.print(embed(inp_arr))
tf.print(embed(inp_arr).shape)

tf.print(embed(np.array([1])))


# 
# ---

# 
# 
# Czytelnicy mogą zignorować poniższą komórkę.
# 




